﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;

namespace OPlanner {

	public partial class PrintAreaProperties : Window {

		private bool isDragging;
		private Point dragStart;
		private OCurrentInfo info;
		public Size areaSize;
		private double xOffset;
		private double yOffset;

		public PrintAreaProperties() {
			InitializeComponent();
			areaSize = Size.Empty;
			isDragging = false;
			printAreaVisual.MouseLeftButtonDown += new MouseButtonEventHandler(printAreaVisual_MouseLeftButtonDown);
			printAreaVisual.MouseLeftButtonUp += new MouseButtonEventHandler(printAreaVisual_MouseLeftButtonUp);
			printAreaVisual.MouseMove += new MouseEventHandler(printAreaVisual_MouseMove);
			printAreaVisual.LostMouseCapture += new MouseEventHandler(printAreaVisual_LostMouseCapture);
			Loaded += new RoutedEventHandler(PrintAreaProperties_Loaded);
		}

		void PrintAreaProperties_Loaded(object sender, RoutedEventArgs e) {
			mainScroll.ScrollToHorizontalOffset(xOffset + mainContent.Width / 2 - mainScroll.ViewportWidth / 2);
			mainScroll.ScrollToVerticalOffset(yOffset + mainContent.Height / 2 - mainScroll.ViewportHeight / 2);
		}

		public void SetInfo(OCurrentInfo newInfo) {
			info = newInfo;
			LoadData();
		}

		private void LoadData() {
			if (info.Data == null) {
				return;
			}
			xOffset = info.Data.CoursesPrintXOffset;
			yOffset = info.Data.CoursesPrintYOffset;
			OCurrentInfo newInfo = new OCurrentInfo();
			mainCanvas.SetInfo(newInfo);
			newInfo.LoadData(info.Data.Clone());
			newInfo.CourseViewType = CourseViewType.Preview;
			newInfo.InvokeCourseViewModeChanged(this);
			mainContent.Width = mainCanvas.CanvasWidth;
			mainContent.Height = mainCanvas.CanvasHeight;
			UpdateUI(true);
		}

		public void SetAreaSize(Size newSize) {
			areaSize = newSize;
			printAreaVisual.Width = areaSize.Width;
			printAreaVisual.Height = areaSize.Height;
		}

		private void saveButton_Click(object sender, RoutedEventArgs e) {
			SaveData();
			Close();
		}

		private void SaveData() {
			if (info == null) {
				return;
			}
			if (info.Data == null) {
				return;
			}
			if (xOffset != info.Data.CoursesPrintXOffset || yOffset != info.Data.CoursesPrintYOffset) {
				info.InvokeDataChanging(this, "Set Print Area");
				info.Data.CoursesPrintXOffset = xOffset;
				info.Data.CoursesPrintYOffset = yOffset;
				info.InvokeDataChanged(this, "Set Print Area");
			}
			Close();
		}

		private void cancelButton_Click(object sender, RoutedEventArgs e) {
			Close();
		}

		private void resetButton_Click(object sender, RoutedEventArgs e) {
			xOffset = 0;
			yOffset = 0;
			UpdateUI(true);
		}

		private void printAreaVisual_LostMouseCapture(object sender, MouseEventArgs e) {
			if (isDragging) {
				Point mousePos = Mouse.GetPosition(mainContent);
				xOffset = mousePos.X - dragStart.X - mainContent.Width / 2 + printAreaVisual.Width / 2;
				yOffset = mousePos.Y - dragStart.Y - mainContent.Height / 2 + printAreaVisual.Height / 2;
				UpdateUI(true);
				isDragging = false;
			}
		}

		private void printAreaVisual_MouseLeftButtonDown(object sender, MouseButtonEventArgs e) {
			if (info == null) {
				return;
			}
			if (info.Data == null) {
				return;
			}
			Point mousePos = Mouse.GetPosition(mainContent);
			dragStart = new Point(mousePos.X - Canvas.GetLeft(printAreaVisual), mousePos.Y - Canvas.GetTop(printAreaVisual));
			isDragging = true;
			Mouse.Capture(printAreaVisual);
		}

		private void printAreaVisual_MouseMove(object sender, MouseEventArgs e) {
			if (isDragging) {
				Point mousePos = Mouse.GetPosition(mainContent);
				xOffset = mousePos.X - dragStart.X - mainContent.Width / 2 + printAreaVisual.Width/2;
				yOffset = mousePos.Y - dragStart.Y - mainContent.Height /2 + printAreaVisual.Height/2;
				UpdateUI(false);
			}
		}

		private void printAreaVisual_MouseLeftButtonUp(object sender, MouseButtonEventArgs e) {
			if (isDragging) {
				Mouse.Capture(null);
			}
		}

		private void UpdateUI(bool updateText) {
			if (updateText) {
				xOffsetText.Text = Convert.ToString(xOffset);
				yOffsetText.Text = Convert.ToString(yOffset);
			}
			Canvas.SetLeft(printAreaVisual, xOffset + mainContent.Width / 2 - printAreaVisual.Width/2);
			Canvas.SetTop(printAreaVisual, yOffset + mainContent.Height / 2-printAreaVisual.Height/2);
		}

		private void xOffsetText_PreviewLostKeyboardFocus(object sender, KeyboardFocusChangedEventArgs e) {
			double num;
			if(double.TryParse(xOffsetText.Text,out num)){
				xOffset = num;
			}
			UpdateUI(true);
		}

		private void yOffsetText_PreviewLostKeyboardFocus(object sender, KeyboardFocusChangedEventArgs e) {
			double num;
			if (double.TryParse(yOffsetText.Text, out num)) {
				yOffset = num;
			}
			UpdateUI(true);
		}

	}
}
