﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;

namespace OcdToXamlConverter {

	public enum SpacingMode {
		Relative, Absolute
	}

	public enum CustomTileMode {
		Normal, Honeycomb
	}

	public enum CustomOffsetMode {
		Predictable, Normal
	}

	public class TiledArea : Control {

		public static DependencyProperty DataProperty;
		public static DependencyProperty HorizontalSpacingProperty;
		public static DependencyProperty VerticalSpacingProperty;
		public static DependencyProperty TargetDrawingProperty;
		public static DependencyProperty FillTransformProperty;
		public static DependencyProperty SpacingModeProperty;
		public static DependencyProperty TileModeProperty;
		public static DependencyProperty OffsetModeProperty;

		static TiledArea() {
			DataProperty = DependencyProperty.Register("Data", typeof(Geometry), typeof(TiledArea));
			HorizontalSpacingProperty = DependencyProperty.Register("HorizontalSpacing", typeof(double), typeof(TiledArea));
			VerticalSpacingProperty = DependencyProperty.Register("VerticalSpacing", typeof(double), typeof(TiledArea));
			TargetDrawingProperty = DependencyProperty.Register("TargetDrawing", typeof(Drawing), typeof(TiledArea));
			FillTransformProperty = DependencyProperty.Register("FillTransform", typeof(Transform), typeof(TiledArea));
			SpacingModeProperty = DependencyProperty.Register("SpacingMode", typeof(SpacingMode), typeof(TiledArea));
			TileModeProperty = DependencyProperty.Register("TileMode", typeof(CustomTileMode), typeof(TiledArea));
			OffsetModeProperty = DependencyProperty.Register("OffsetMode", typeof(CustomOffsetMode), typeof(TiledArea));
		}

		public Geometry Data {
			get {
				return (Geometry)base.GetValue(DataProperty);
			}
			set {
				if (Data != value) {
					InvalidateVisual();
				}
				base.SetValue(DataProperty, value);
			}
		}

		public double HorizontalSpacing {
			get {
				return (double)base.GetValue(HorizontalSpacingProperty);
			}
			set {
				base.SetValue(HorizontalSpacingProperty, value);
			}
		}

		public double VerticalSpacing {
			get {
				return (double)base.GetValue(VerticalSpacingProperty);
			}
			set {
				base.SetValue(VerticalSpacingProperty, value);
			}
		}

		public Drawing TargetDrawing {
			get {
				return (Drawing)base.GetValue(TargetDrawingProperty);
			}
			set {
				base.SetValue(TargetDrawingProperty, value);
			}
		}

		public Transform FillTransform {
			get {
				return (Transform)base.GetValue(FillTransformProperty);
			}
			set {
				base.SetValue(FillTransformProperty, value);
			}
		}

		public SpacingMode SpacingMode {
			get {
				return (SpacingMode)base.GetValue(SpacingModeProperty);
			}
			set {
				base.SetValue(SpacingModeProperty, value);
			}
		}

		public CustomTileMode TileMode {
			get {
				return (CustomTileMode)base.GetValue(TileModeProperty);
			}
			set {
				base.SetValue(TileModeProperty, value);
			}
		}

		public CustomOffsetMode OffsetMode {
			get {
				return (CustomOffsetMode)base.GetValue(OffsetModeProperty);
			}
			set {
				base.SetValue(OffsetModeProperty, value);
			}
		}

		protected override void OnRender(DrawingContext drawingContext) {
			if (TargetDrawing == null || Data == null) {
				return;
			}
			drawingContext.PushClip(Data);
			Rect rect = Data.Bounds;
			Rect newRect = rect;
			if (FillTransform != null) {
				GeneralTransform inverse = FillTransform.Inverse;
				if (inverse != null) {
					if (inverse is Transform) {
						Geometry transformedData = Data.Clone();
						transformedData.Transform = (Transform)inverse;
						newRect = transformedData.Bounds;
					} else {
						newRect = inverse.TransformBounds(rect);
					}
				}
			}
			Rect targetRect = TargetDrawing.Bounds;
			double width;
			if (SpacingMode == SpacingMode.Absolute && HorizontalSpacing > 0) {
				width = HorizontalSpacing;
			} else {
				width = targetRect.Width + HorizontalSpacing;
				if (width <= 0) {
					width = targetRect.Width;
				}
			}
			double height;
			if (SpacingMode == SpacingMode.Absolute && VerticalSpacing > 0) {
				height = VerticalSpacing;
			} else {
				height = targetRect.Height + VerticalSpacing;
				if (height <= 0) {
					height = targetRect.Height;
				}
			}
			if (FillTransform != null) {
				drawingContext.PushTransform(FillTransform);
			}
			double xOffset = -targetRect.Left;
			double yOffset = -targetRect.Top;
			double extraXOffset = 0;
			double extraYOffset = 0;
			if (OffsetMode == CustomOffsetMode.Predictable) {
				int num;
				num = (int)Math.Round(newRect.Left / width);
				extraXOffset = width * num - newRect.Left;
				num = (int)Math.Round(newRect.Top / height);
				extraYOffset = height * num - newRect.Top;
			}
			int row = 0;
			for (double yPos = newRect.Top+extraYOffset; yPos <= newRect.Bottom; yPos += height) {
				double extraOffset = 0;
				if (row == 1 && TileMode == CustomTileMode.Honeycomb) {
					extraOffset = -width / 2;
				}
				for (double xPos = newRect.Left+extraOffset+extraXOffset; xPos <= newRect.Right; xPos += width) {
					Rect currentBounds = new Rect(xPos + xOffset, yPos + yOffset, targetRect.Width, targetRect.Height);
					if (FillTransform != null) {
						currentBounds = FillTransform.TransformBounds(currentBounds);
					}
					if (Rect.Intersect(currentBounds, rect) == Rect.Empty) {
						continue;
					}
					TranslateTransform translate = new TranslateTransform(xPos + xOffset, yPos + yOffset);
					drawingContext.PushTransform(translate);
					drawingContext.DrawDrawing(TargetDrawing);
					drawingContext.Pop();
				}
				row++;
				if (row >= 2) {
					row = 0;
				}
			}
			if (FillTransform != null) {
				drawingContext.Pop();
			}
			drawingContext.Pop();
		}

	}

	public class HatchedArea : Control {

		public static DependencyProperty DataProperty;
		public static DependencyProperty HatchLineWidthProperty;
		public static DependencyProperty HatchLineSpacingProperty;
		public static DependencyProperty HatchColorProperty;
		public static DependencyProperty FillTransformProperty;
		public static DependencyProperty SpacingModeProperty;
		public static DependencyProperty OffsetModeProperty;

		static HatchedArea() {
			DataProperty = DependencyProperty.Register("Data", typeof(Geometry), typeof(HatchedArea));
			HatchLineWidthProperty = DependencyProperty.Register("HatchLineWidth", typeof(double), typeof(HatchedArea));
			HatchLineSpacingProperty = DependencyProperty.Register("HatchLineSpacing", typeof(double), typeof(HatchedArea));
			HatchColorProperty = DependencyProperty.Register("HatchColor", typeof(Color), typeof(HatchedArea));
			FillTransformProperty = DependencyProperty.Register("FillTransform", typeof(Transform), typeof(HatchedArea));
			SpacingModeProperty = DependencyProperty.Register("SpacingMode", typeof(SpacingMode), typeof(HatchedArea));
			OffsetModeProperty = DependencyProperty.Register("OffsetMode", typeof(CustomOffsetMode), typeof(HatchedArea));
		}

		public Geometry Data {
			get {
				return (Geometry)base.GetValue(DataProperty);
			}
			set {
				if (Data != value) {
					InvalidateVisual();
				}
				base.SetValue(DataProperty, value);
			}
		}

		public double HatchLineWidth {
			get {
				return (double)base.GetValue(HatchLineWidthProperty);
			}
			set {
				base.SetValue(HatchLineWidthProperty, value);
			}
		}

		public double HatchLineSpacing {
			get {
				return (double)base.GetValue(HatchLineSpacingProperty);
			}
			set {
				base.SetValue(HatchLineSpacingProperty, value);
			}
		}

		public Color HatchColor {
			get {
				return (Color)base.GetValue(HatchColorProperty);
			}
			set {
				base.SetValue(HatchColorProperty, value);
			}
		}

		public Transform FillTransform {
			get {
				return (Transform)base.GetValue(FillTransformProperty);
			}
			set {
				base.SetValue(FillTransformProperty, value);
			}
		}

		public SpacingMode SpacingMode {
			get {
				return (SpacingMode)base.GetValue(SpacingModeProperty);
			}
			set {
				base.SetValue(SpacingModeProperty, value);
			}
		}

		public CustomOffsetMode OffsetMode {
			get {
				return (CustomOffsetMode)base.GetValue(OffsetModeProperty);
			}
			set {
				base.SetValue(OffsetModeProperty, value);
			}
		}

		protected override void OnRender(DrawingContext drawingContext) {
			if (HatchLineWidth <= 0 || Data == null) {
				return;
			}
			drawingContext.PushClip(Data);
			Rect rect = Data.Bounds;
			Rect newRect = rect;
			if (FillTransform != null) {
				if (FillTransform.Inverse != null) {
					newRect = FillTransform.Inverse.TransformBounds(rect);
				}
			}
			double width;
			if (SpacingMode == SpacingMode.Absolute && HatchLineSpacing > 0) {
				width = HatchLineSpacing;
			} else {
				width = HatchLineWidth + HatchLineSpacing;
				if (width <= 0) {
					return;
				}
			}
			if (FillTransform != null) {
				drawingContext.PushTransform(FillTransform);
			}
			Pen pen = new Pen(new SolidColorBrush(HatchColor), HatchLineWidth);
			double extraXOffset = 0;
			if (OffsetMode == CustomOffsetMode.Predictable) {
				int num = (int)Math.Round(newRect.Left / width);
				extraXOffset = width * num - newRect.Left;
			}
			for (double xPos = newRect.Left + extraXOffset; xPos <= newRect.Right + HatchLineWidth / 2; xPos += width) {
				drawingContext.DrawLine(pen, new Point(xPos, newRect.Top), new Point(xPos, newRect.Bottom));
			}
			if (FillTransform != null) {
				drawingContext.Pop();
			}
			drawingContext.Pop();
		}

	}

	public class CustomDrawingContainer : Control {

		public static DependencyProperty DrawingProperty;

		static CustomDrawingContainer() {
			DrawingProperty = DependencyProperty.Register("Drawing", typeof(Drawing), typeof(CustomDrawingContainer));
		}

		public Drawing Drawing {
			get {
				return (Drawing)base.GetValue(DrawingProperty);
			}
			set {
				if (Drawing != value) {
					InvalidateVisual();
				}
				base.SetValue(DrawingProperty, value);
			}
		}

		protected override void OnRender(DrawingContext drawingContext) {
			if (Drawing != null) {
				drawingContext.DrawDrawing(Drawing);
			}
		}

	}

}