﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Windows.Markup;

namespace OcdToXamlConverter {

	public abstract partial class OcdBaseFileReader {

		protected class OcdPathHelper {

			public static List<TCord[]> SplitCordsByCorner(TCord[] points) {
				List<TCord[]> pointsList = new List<TCord[]>();
				List<TCord> currentPoints = new List<TCord>();
				if (points.Length == 0) {
					return pointsList;
				}
				for (int i = 0; i < points.Length; i++) {
					currentPoints.Add(points[i]);
					if (points[i].IsCornerPoint && i > 0) {
						pointsList.Add(currentPoints.ToArray());
						currentPoints = new List<TCord>();
						if (i < points.Length - 1) {
							currentPoints.Add(points[i]);
						}
					}
				}
				if (currentPoints.Count > 0) {
					pointsList.Add(currentPoints.ToArray());
				}
				return pointsList;
			}

			public List<OcdPathHelper> SplitByCorner() {
				List<OcdPathHelper> paths = new List<OcdPathHelper>();
				List<TCord[]> split = SplitCordsByCorner(points);
				for (int i = 0; i < split.Count; i++) {
					paths.Add(new OcdPathHelper(split[i]));
				}
				return paths;
			}

			TCord[] points;
			List<OcdBaseSegment> ocdSegments;
			public List<OcdBaseSegment> OcdSegments {
				get {
					return ocdSegments;
				}
			}

			double length;

			public OcdPathHelper(TCord[] points) {
				this.points = points;
				LoadPoints();
				length = GetLength();
			}

			private double GetLength() {
				double val = 0;
				for (int i = 0; i < ocdSegments.Count; i++) {
					val += ocdSegments[i].Length;
				}
				return val;
			}

			public Point Start {
				get {
					if (points.Length == 0) {
						return new Point(0, 0);
					}
					return points[0].GetPoint();
				}
			}

			public Point End {
				get {
					if (points.Length == 0) {
						return new Point(0, 0);
					}
					return points[points.Length-1].GetPoint();
				}
			}

			public Point GetPointAtDistance(double distance) {
				if (distance <= 0) {
					return Start;
				}
				if (distance >= Length) {
					return End;
				}
				double val = 0;
				double current = 0;
				for (int i = 0; i < ocdSegments.Count; i++) {
					current = ocdSegments[i].Length;
					if (val + current >= distance) {
						return ocdSegments[i].GetPointAtDistance(distance - val);
					}
					val += current;
				}
				return End;
			}

			public Vector GetTangentAtDistance(double distance) {
				if (ocdSegments.Count == 0) {
					Console.WriteLine("Get tangent error");
					return new Vector(1, 0);
				}
				if (distance <= 0) {
					return ocdSegments[0].GetTangentAtStart(false);
				}
				if (distance >= Length) {
					return ocdSegments[ocdSegments.Count-1].GetTangentAtEnd(false);
				}
				double val = 0;
				double current = 0;
				for (int i = 0; i < ocdSegments.Count; i++) {
					current = ocdSegments[i].Length;
					if (val + current >= distance) {
						return ocdSegments[i].GetTangentAtDistance(distance - val);
					}
					val += current;
				}
				return ocdSegments[ocdSegments.Count-1].GetTangentAtEnd(false);
			}

			public double GetAngleAtDistance(double distance) {
				Vector tangent = GetTangentAtDistance(distance);
				if (tangent.Length == 0) {
					Console.WriteLine("Get angle error");
					return 0;
				}
				return Math.Atan2(tangent.Y, tangent.X);
			}

			public double Length {
				get { return length; }
			}

			private void LoadPoints(){
				ocdSegments = new List<OcdBaseSegment>();
				if (points.Length == 0) {
					return;
				}
				TCord lastPoint = points[0];
				OcdBaseSegment prevSegment = null;
				for (int i = 1; i < points.Length; i++) {
					if (points[i].IsFirstCurvePoint) {
						if (i >= points.Length - 2) {
							break;
						}
						OcdBezierSegment segment = new OcdBezierSegment(lastPoint.GetPoint(), points[i].GetPoint(), points[i + 1].GetPoint(), points[i + 2].GetPoint());
						segment.IsStartCornerPoint = lastPoint.IsCornerPoint;
						segment.IsEndCornerPoint = points[i + 2].IsCornerPoint;
						segment.IsLeftStroked = lastPoint.IsLeftStroked;
						segment.IsRightStroked = lastPoint.IsRightStroked;
						if (prevSegment!=null) {
							segment.PreviousSegment = prevSegment;
							prevSegment.NextSegment = segment;
						}
						ocdSegments.Add(segment);
						lastPoint = points[i + 2];
						i += 2;
						prevSegment = segment;
					} else {
						OcdLineSegment segment = new OcdLineSegment(lastPoint.GetPoint(), points[i].GetPoint());
						segment.IsStartCornerPoint = lastPoint.IsCornerPoint;
						segment.IsEndCornerPoint = points[i].IsCornerPoint;
						segment.IsLeftStroked = lastPoint.IsLeftStroked;
						segment.IsRightStroked = lastPoint.IsRightStroked;
						if (prevSegment != null) {
							segment.PreviousSegment = prevSegment;
							prevSegment.NextSegment = segment;
						}
						ocdSegments.Add(segment);
						lastPoint = points[i];
						prevSegment = segment;
					}
				}
			}

			public PathFigure GetPathFigure() {
				if (points.Length == 0) {
					return null;
				}
				PathFigure pathFigure = new PathFigure();
				PathSegmentCollection pathSegmentCollection = new PathSegmentCollection();
				pathFigure.Segments = pathSegmentCollection;
				pathFigure.StartPoint = points[0].GetPoint();
				for (int i = 1; i < points.Length; i++) {
					if (points[i].IsFirstCurvePoint) {
						if (i >= points.Length - 2) {
							break;
						}
						BezierSegment bezierSegment = new BezierSegment(points[i].GetPoint(), points[i+1].GetPoint(), points[i+2].GetPoint(), true);
						pathSegmentCollection.Add(bezierSegment);
						i += 2;
					} else {
						LineSegment lineSegment = new LineSegment();
						lineSegment.Point = points[i].GetPoint();
						pathSegmentCollection.Add(lineSegment);
					}
				}
				return pathFigure;
			}

			public PathFigure GetShiftedPath(double distance) {
				PathFigure newFigure = new PathFigure();
				PathSegmentCollection segments = new PathSegmentCollection();
				newFigure.Segments = segments;
				bool started = false;
				for (int i = 0; i < ocdSegments.Count; i++) {
					OcdBaseSegment[] newSegments = ocdSegments[i].Shift(distance);
					if (newSegments.Length == 0) {
						continue;
					}
					if (started == false) {
						newFigure.StartPoint = newSegments[0].Start;
						started = true;
					}
					for (int j = 0; j < newSegments.Length; j++) {
						segments.Add(newSegments[j].PathSegment);
					}
				}
				return newFigure;
			}

			private static OcdBaseSegment GetSegmentAtDistance(List<OcdBaseSegment> segments, double distance, out double remainingDistance, out int segmentNum) {
				double val = 0;
				double current = 0;
				remainingDistance = 0;
				for (int i = 0; i < segments.Count; i++) {
					current = segments[i].Length;
					if (val + current >= distance) {
						remainingDistance = distance - val;
						segmentNum = i;
						return segments[i];
					}
					val += current;
				}
				segmentNum = -1;
				return null;
			}

			private static List<OcdBaseSegment> ReverseSegments(List<OcdBaseSegment> segments) {
				List<OcdBaseSegment> newSegments = new List<OcdBaseSegment>();
				for (int i = 0; i < segments.Count; i++) {
					newSegments.Add(segments[segments.Count - i - 1].Reverse());
				}
				for (int i = 0; i < newSegments.Count; i++) {
					if (i > 0) {
						newSegments[i].PreviousSegment = newSegments[i - 1];
					}
					if (i < newSegments.Count - 1) {
						newSegments[i].NextSegment = newSegments[i + 1];
					}
				}
				return newSegments;
			}

			public PathFigure GetPointedPath(double width, double startDistance, double endDistance) {
				double newStartDistance = startDistance;
				double newEndDistance = endDistance;
				if (startDistance + endDistance > Length-0.00001) {
					newStartDistance = startDistance * Length / (startDistance + endDistance + 0.0001);
					newEndDistance = endDistance * Length / (startDistance + endDistance + 0.0001);
				}
				PathFigure newFigure = new PathFigure();
				PathSegmentCollection pathSegments = new PathSegmentCollection();
				newFigure.Segments = pathSegments;
				List<OcdBaseSegment> newSegments = new List<OcdBaseSegment>(ocdSegments);
				double remaining;
				int num;
				OcdBaseSegment segment = GetSegmentAtDistance(newSegments, newStartDistance, out remaining, out num);
				if (segment != null) {
					OcdBaseSegment[] segments = segment.Split(remaining);
					newSegments.Remove(segment);
					for (int i = 0; i < segments.Length; i++) {
						newSegments.Insert(i + num, segments[i]);
					}
				}
				segment = GetSegmentAtDistance(newSegments, Length - newEndDistance, out remaining, out num);
				if (segment != null) {
					OcdBaseSegment[] segments = segment.Split(remaining);
					newSegments.Remove(segment);
					for (int i = 0; i < segments.Length; i++) {
						newSegments.Insert(i + num, segments[i]);
					}
				}
				newFigure.StartPoint = Start;
				double distance = 0;
				double oldShiftDistance = 0;
				double shiftDistance = 0;
				for (int i = 0; i < newSegments.Count; i++) {
					distance += newSegments[i].Length;
					if (distance < newStartDistance) {
						shiftDistance = distance / newStartDistance * (width / 2);
					} else if (distance > Length - newEndDistance) {
						shiftDistance = (Length - distance) / newEndDistance * (width / 2);
					} else {
						shiftDistance = width / 2;
					}
					OcdBaseSegment[] shiftedSegments = newSegments[i].Shift(oldShiftDistance, shiftDistance);
					for (int j = 0; j < shiftedSegments.Length; j++) {
						pathSegments.Add(shiftedSegments[j].PathSegment);
					}
					oldShiftDistance = shiftDistance;
				}
				List<OcdBaseSegment> reversedSegments = ReverseSegments(newSegments);
				distance = 0;
				oldShiftDistance = 0;
				shiftDistance = 0;
				for (int i = 0; i < reversedSegments.Count; i++) {
					distance += reversedSegments[i].Length;
					if (distance < newEndDistance) {
						shiftDistance = distance / newEndDistance * (width / 2);
					} else if (distance > Length - newStartDistance) {
						shiftDistance = (Length - distance) / newStartDistance * (width / 2);
					} else {
						shiftDistance = width / 2;
					}
					OcdBaseSegment[] shiftedSegments = reversedSegments[i].Shift(oldShiftDistance, shiftDistance);
					for (int j = 0; j < shiftedSegments.Length; j++) {
						pathSegments.Add(shiftedSegments[j].PathSegment);
					}
					oldShiftDistance = shiftDistance;
				}
				return newFigure;
			}

		}

		protected abstract class OcdBaseSegment {

			public OcdBaseSegment() {
				isStroked = true;
				isLeftStroked = true;
				isRightStroked = true;
				isEndCornerPoint = false;
				isStartCornerPoint = false;
			}

			public abstract double Length {
				get;
			}

			bool isStroked;
			public bool IsStroked {
				get {
					return isStroked;
				}
				set {
					isStroked = value;
				}
			}
			bool isLeftStroked;
			public bool IsLeftStroked {
				get {
					return isLeftStroked;
				}
				set {
					isLeftStroked = value;
				}
			}
			bool isRightStroked;
			public bool IsRightStroked {
				get {
					return isRightStroked;
				}
				set {
					isRightStroked = value;
				}
			}

			bool isEndCornerPoint;
			public bool IsEndCornerPoint {
				get {
					return isEndCornerPoint;
				}
				set {
					isEndCornerPoint = value;
				}
			}

			bool isStartCornerPoint;
			public bool IsStartCornerPoint {
				get {
					return isStartCornerPoint;
				}
				set {
					isStartCornerPoint = value;
				}
			}

			public Point Start {
				get {
					return start;
				}
			}

			public Point End {
				get {
					return end;
				}
			}
			protected Point start;
			protected Point end;

			public abstract Point GetPointAtDistance(double distance);

			public abstract Vector GetTangentAtDistance(double distance);

			public abstract Vector GetTangentAtEnd(bool average);

			public abstract Vector GetTangentAtStart(bool average);

			public double GetAngleAtDistance(double distance) {
				Vector tangent = GetTangentAtDistance(distance);
				return Math.Atan2(tangent.Y, tangent.X);
			}

			public double GetAngleAtEnd(bool average) {
				Vector tangent = GetTangentAtEnd(average);
				return Math.Atan2(tangent.Y, tangent.X);
			}

			public double GetAngleAtStart(bool average) {
				Vector tangent = GetTangentAtStart(average);
				return Math.Atan2(tangent.Y, tangent.X);
			}

			OcdBaseSegment previousSegment;
			public OcdBaseSegment PreviousSegment {
				get {
					return previousSegment;
				}
				set {
					previousSegment = value;
				}
			}
			OcdBaseSegment nextSegment;
			public OcdBaseSegment NextSegment {
				get {
					return nextSegment;
				}
				set {
					nextSegment = value;
				}
			}

			public abstract PathSegment PathSegment {
				get;
			}

			public abstract OcdBaseSegment Clone();

			public abstract OcdBaseSegment[] Split(double distance);

			public abstract OcdBaseSegment[] Shift(double distance);

			public abstract OcdBaseSegment[] Shift(double startDistance, double endDistance);

			public abstract OcdBaseSegment Reverse();

		}

		protected class OcdLineSegment : OcdBaseSegment {

			public OcdLineSegment(Point start, Point end) {
				this.start = start;
				this.end = end;
			}

			public override PathSegment PathSegment {
				get {
					return new LineSegment(End, IsStroked);
				}
			}

			public override double Length {
				get { return Point.Subtract(End, Start).Length; }
			}

			public override Point GetPointAtDistance(double distance) {
				if (distance < 0) {
					return Start;
				}
				if (distance > Length) {
					return End;
				}
				if (Length == 0) {
					return Start;
				}
				return Point.Add(Start, Vector.Multiply(distance / Length, Point.Subtract(End, Start)));
			}

			public override Vector GetTangentAtDistance(double distance) {
				Vector dir = Point.Subtract(End, Start);
				if (dir.Length == 0) {
					Console.WriteLine("Get tangent error");
					return new Vector(1, 0);
				}
				dir.Normalize();
				return dir;
			}

			public override Vector GetTangentAtEnd(bool average) {
				Vector dir = Point.Subtract(End, Start);
				if (dir.Length == 0) {
					Console.WriteLine("Get tangent error");
					return new Vector(1, 0);
				}
				dir.Normalize();
				if (average == false) {
					return dir;
				}
				if (NextSegment == null) {
					return dir;
				}
				Vector sum = Vector.Add(dir, NextSegment.GetTangentAtStart(false));
				if (sum.Length == 0) {
					Console.WriteLine("Get tangent error");
					return new Vector(1, 0);
				}
				sum.Normalize();
				return sum;
			}

			public override Vector GetTangentAtStart(bool average) {
				Vector dir = Point.Subtract(End, Start);
				if (dir.Length == 0) {
					Console.WriteLine("Get tangent error");
					return new Vector(1, 0);
				}
				dir.Normalize();
				if (average == false) {
					return dir;
				}
				if (PreviousSegment == null) {
					return dir;
				}
				Vector sum = Vector.Add(dir, PreviousSegment.GetTangentAtEnd(false));
				if (sum.Length == 0) {
					Console.WriteLine("Get tangent error");
					return new Vector(1, 0);
				}
				sum.Normalize();
				return sum;
			}

			public override OcdBaseSegment Clone() {
				return Clone(start, end);
			}

			private OcdBaseSegment Clone(Point start, Point end) {
				OcdLineSegment segment = new OcdLineSegment(start, end);
				segment.IsStroked = IsStroked;
				segment.IsEndCornerPoint = IsEndCornerPoint;
				segment.IsStartCornerPoint = IsStartCornerPoint;
				segment.IsLeftStroked = IsLeftStroked;
				segment.IsRightStroked = IsRightStroked;
				return segment;
			}

			public override OcdBaseSegment[] Split(double distance) {
				if (distance <= 0 || distance >= Length) {
					return new OcdBaseSegment[] { Clone() };
				}
				Point point = GetPointAtDistance(distance);
				OcdBaseSegment segment1 = Clone(start, point);
				OcdBaseSegment segment2 = Clone(point, end);
				return new OcdBaseSegment[] { segment1, segment2 };
			}

			public override OcdBaseSegment[] Shift(double distance) {
				return Shift(distance, distance);
			}

			public override OcdBaseSegment[] Shift(double startDistance, double endDistance) {
				double averageAngle = GetAngleAtStart(true);
				double differenceAverageAngle;
				if (PreviousSegment != null) {
					differenceAverageAngle = (PreviousSegment.GetAngleAtEnd(false) - GetAngleAtStart(false)) / 2;
				} else {
					differenceAverageAngle = 0;
				}
				double cosAngle = Math.Abs(Math.Cos(differenceAverageAngle));//
				if (cosAngle < 0.01) {
					cosAngle = 0.01;
				}
				double distanceFromLine = startDistance / cosAngle;
				Point newStart = new Point(Start.X + Math.Sin(averageAngle) * distanceFromLine, Start.Y - Math.Cos(averageAngle) * distanceFromLine);
				averageAngle = GetAngleAtEnd(true);
				if (NextSegment != null) {
					differenceAverageAngle = (GetAngleAtEnd(false) - NextSegment.GetAngleAtStart(false)) / 2;
				} else {
					differenceAverageAngle = 0;
				}
				cosAngle = Math.Abs(Math.Cos(differenceAverageAngle));//
				if (cosAngle < 0.01) {
					cosAngle = 0.01;
				}
				distanceFromLine = endDistance / cosAngle;
				Point newEnd = new Point(End.X + Math.Sin(averageAngle) * distanceFromLine, End.Y - Math.Cos(averageAngle) * distanceFromLine);
				OcdBaseSegment newSegment = Clone(newStart, newEnd);
				if (startDistance >= 0) {
					newSegment.IsStroked = IsLeftStroked;
				} else {
					newSegment.IsStroked = IsRightStroked;
				}
				return new OcdBaseSegment[]{newSegment};
			}

			public override OcdBaseSegment Reverse() {
				return Clone(end, start);
			}

		}

		protected class OcdBezierSegment : OcdBaseSegment {

			private Vector startVector {
				get {
					return new Vector(Start.X, Start.Y);
				}
			}
			private Vector control1Vector {
				get {
					return new Vector(Control1.X, Control1.Y);
				}
			}
			private Vector control2Vector {
				get {
					return new Vector(Control2.X, Control2.Y);
				}
			}
			private Vector endVector {
				get {
					return new Vector(End.X, End.Y);
				}
			}
			double length;
			Point control1;
			Point control2;
			public Point Control1 {
				get {
					return control1;
				}
			}
			public Point Control2 {
				get {
					return control2;
				}
			}

			public OcdBezierSegment(Point start, Point control1, Point control2, Point end) {
				this.start = start;
				this.end = end;
				this.control1 = control1;
				this.control2 = control2;
				length = GetLength();
			}

			private double GetLength() {
				return GetPartialLength(1);
			}

			private Point GetPointAtT(double t) {
				if (t <= 0) {
					return Start;
				}
				if (t >= 1) {
					return End;
				}
				Vector vector = Vector.Multiply(Math.Pow(1 - t, 3), startVector) + Vector.Multiply(3 * Math.Pow(1 - t, 2) * t, control1Vector) + Vector.Multiply(3 * Math.Pow(t, 2) * (1 - t), control2Vector) + Vector.Multiply(Math.Pow(t, 3), endVector);
				return new Point(vector.X, vector.Y);
			}

			private double GetPartialLength(double t) {
				Point endPoint = GetPointAtT(t);
				int numIntervals = (int)Math.Min(100, Math.Max(10, Point.Subtract(Start, endPoint).Length / 10));
				double length = 0;
				double width = t / ((double)numIntervals);
				Point point = Start;
				for (int i = 1; i <= numIntervals; i++) {
					Point newPoint = GetPointAtT(i * width);
					length += Point.Subtract(point, newPoint).Length;
					point = newPoint;
				}
				return length;
			}

			public override PathSegment PathSegment {
				get {
					return new BezierSegment(Control1, Control2, End, IsStroked);
				}
			}

			private double GetTAtDistance(double distance) {
				if (distance <= 0) {
					return 0;
				}
				if (distance >= Length) {
					return 1;
				}
				if (Length == 0) {
					return 0;
				}
				double t;
				double approxDist;
				t = distance / Length;
				approxDist = GetPartialLength(t);
				if (approxDist < distance) {
					return GetTAtDistanceRecursive(distance, t, 1.0, 15);
				}
				if (approxDist > distance) {
					return GetTAtDistanceRecursive(distance, 0, t, 15);
				}
				return t;
			}

			private double GetTAtDistanceRecursive(double distance, double lowerT, double upperT, int maxRecursions) {
				double midT = (upperT + lowerT) / 2;
				double t = midT;
				if (maxRecursions <= 0) {
					return t;
				}
				double approxDist;
				approxDist = GetPartialLength(t);
				if (approxDist < distance) {
					return GetTAtDistanceRecursive(distance, midT, upperT, maxRecursions - 1);
				}
				if (approxDist > distance) {
					return GetTAtDistanceRecursive(distance, lowerT, midT, maxRecursions - 1);
				}
				return t;
			}

			public override double Length {
				get { return length; }
			}

			public override Point GetPointAtDistance(double distance) {
				if (distance < 0) {
					return Start;
				}
				if (distance > Length) {
					return End;
				}
				if (Length == 0) {
					return Start;
				}
				double t = GetTAtDistance(distance);
				return GetPointAtT(t);
			}

			private Vector GetTangentAtT(double t) {
				Vector derivative = new Vector(0, 0);
				derivative = Vector.Multiply(-3 * Math.Pow(1 - t, 2), startVector) + Vector.Multiply(-6 * (1 - t) * t + 3 * Math.Pow(1 - t, 2), control1Vector) + Vector.Multiply(6 * (1 - t) * t - 3 * Math.Pow(t, 2), control2Vector) + Vector.Multiply(3 * Math.Pow(t, 2), endVector);
				if (derivative.Length > 0) {
					derivative.Normalize();
				}
				if (derivative.Length == 0) {
					Console.WriteLine("Get tangent error");
					return new Vector(1, 0);
				}
				return derivative;
			}

			public override Vector GetTangentAtDistance(double distance) {
				double t = GetTAtDistance(distance);
				return GetTangentAtT(t);
			}

			public override Vector GetTangentAtEnd(bool average) {
				Vector dir = Point.Subtract(End, Control2);
				if (dir.Length == 0) {
					Console.WriteLine("Get tangent error");
					return new Vector(1, 0);
				}
				dir.Normalize();
				if (average == false) {
					return dir;
				}
				if (NextSegment == null) {
					return dir;
				}
				Vector sum = Vector.Add(dir, NextSegment.GetTangentAtStart(false));
				sum.Normalize();
				return sum;
			}

			public override Vector GetTangentAtStart(bool average) {
				Vector dir = Point.Subtract(Control1, Start);
				if (dir.Length == 0) {
					Console.WriteLine("Get tangent error");
					return new Vector(1, 0);
				}
				dir.Normalize();
				if (average == false) {
					return dir;
				}
				if (PreviousSegment == null) {
					return dir;
				}
				Vector sum = Vector.Add(dir, PreviousSegment.GetTangentAtEnd(false));
				sum.Normalize();
				return sum;
			}

			public override OcdBaseSegment Clone() {
				return Clone(start, control1, control2, end);
			}

			private OcdBaseSegment Clone(Point start, Point control1, Point control2, Point end) {
				OcdBezierSegment segment = new OcdBezierSegment(start, control1, control2, end);
				segment.IsStroked = IsStroked;
				segment.IsEndCornerPoint = IsEndCornerPoint;
				segment.IsStartCornerPoint = IsStartCornerPoint;
				segment.IsLeftStroked = IsLeftStroked;
				segment.IsRightStroked = IsRightStroked;
				return segment;
			}

			private Vector a0 {
				get {
					return startVector;
				}
			}
			private Vector a1 {
				get {
					return Vector.Add(Vector.Multiply(-3,startVector), Vector.Multiply(3,control1Vector));
				}
			}
			private Vector a2 {
				get {
					return Vector.Add(Vector.Add(Vector.Multiply(3, startVector), Vector.Multiply(-6, control1Vector)), Vector.Multiply(3, control2Vector));
				}
			}
			private Vector a3 {
				get {
					return Vector.Add(Vector.Add(Vector.Add(Vector.Multiply(-1, startVector), Vector.Multiply(3, control1Vector)), Vector.Multiply(-3, control2Vector)), endVector);
				}
			}

			private Vector GetP3(Vector a0, Vector a1, Vector a2, Vector a3) {
				return Vector.Multiply(1, Vector.Add(Vector.Add(Vector.Add(a0, a1), a2), a3));
			}

			private Vector GetP2(Vector a0, Vector a1, Vector a2, Vector a3) {
				return Vector.Multiply(1.0/3.0, Vector.Add(Vector.Add(Vector.Multiply(3,a0), Vector.Multiply(2,a1)), a2));
			}

			private Vector GetP1(Vector a0, Vector a1, Vector a2, Vector a3) {
				return Vector.Multiply(1.0 / 3.0, Vector.Add(Vector.Multiply(3, a0), Vector.Multiply(1, a1)));
			}

			private Vector GetP0(Vector a0, Vector a1, Vector a2, Vector a3) {
				return a0;
			}

			public override OcdBaseSegment[] Split(double distance) {
				if (distance <= 0 || distance >= Length) {
					return new OcdBaseSegment[] { Clone() };
				}
				double t = GetTAtDistance(distance);
				OcdBaseSegment segment1;
				OcdBaseSegment segment2;
				Vector newA0;
				Vector newA1;
				Vector newA2;
				Vector newA3;
				Vector p0, p1, p2, p3;
				newA0 = Vector.Multiply(a0, 1);
				newA1 = Vector.Multiply(a1, t);
				newA2 = Vector.Multiply(a2, t*t);
				newA3 = Vector.Multiply(a3, t*t*t);
				p0 = GetP0(newA0, newA1, newA2, newA3);
				p1 = GetP1(newA0, newA1, newA2, newA3);
				p2 = GetP2(newA0, newA1, newA2, newA3);
				p3 = GetP3(newA0, newA1, newA2, newA3);
				segment1 = Clone(new Point(p0.X, p0.Y), new Point(p1.X, p1.Y), new Point(p2.X, p2.Y), new Point(p3.X, p3.Y));
				newA0 = Vector.Add(Vector.Add(Vector.Add(a0, Vector.Multiply(t, a1)), Vector.Multiply(t * t, a2)), Vector.Multiply(t * t * t, a3));
				newA1 = Vector.Multiply(1-t,Vector.Add(Vector.Add(a1, Vector.Multiply(2*t, a2)), Vector.Multiply(3* t * t, a3)));
				newA2 = Vector.Multiply((1 - t)*(1-t), Vector.Add(a2, Vector.Multiply(3 * t, a3)));
				newA3 = Vector.Multiply(a3, (1-t)*(1-t)*(1-t));
				p0 = GetP0(newA0, newA1, newA2, newA3);
				p1 = GetP1(newA0, newA1, newA2, newA3);
				p2 = GetP2(newA0, newA1, newA2, newA3);
				p3 = GetP3(newA0, newA1, newA2, newA3);
				segment2 = Clone(new Point(p0.X, p0.Y), new Point(p1.X, p1.Y), new Point(p2.X, p2.Y), new Point(p3.X, p3.Y));
				return new OcdBaseSegment[] { segment1, segment2 };
			}

			public override OcdBaseSegment[] Shift(double distance) {
				return Shift(distance, distance);
			}

			public override OcdBaseSegment[] Shift(double startDistance, double endDistance) {
				OcdLineSegment[] segments = Flatten();
				OcdBaseSegment[] newSegments = new OcdBaseSegment[segments.Length];
				double distance = 0;
				double newEndDistance;
				for(int i=0;i<segments.Length;i++){
					distance += segments[i].Length;
					if (endDistance != startDistance) {
						newEndDistance = startDistance + distance / Length * (endDistance-startDistance);
					} else {
						newEndDistance = startDistance;
					}
					newSegments[i] = segments[i].Shift(startDistance,newEndDistance)[0];
				}
				return newSegments;
			}

			public OcdLineSegment[] Flatten() {
				int numIntervals = (int)Math.Min(100, Math.Max(20, Length / 4));
				OcdLineSegment[] segments = new OcdLineSegment[numIntervals];
				double interval = 1.0/numIntervals;
				Point lastPoint = Start;
				Point point;
				OcdBaseSegment prevSegment = null;
				for (int i = 1; i <= numIntervals; i++) {
					point = GetPointAtT(interval*i);
					OcdLineSegment segment = new OcdLineSegment(lastPoint, point);
					segments[i-1] = segment;
					lastPoint = point;
					if (prevSegment != null) {
						segment.PreviousSegment = prevSegment;
						prevSegment.NextSegment = segment;
					}
					prevSegment = segment;
				}
				segments[0].PreviousSegment = PreviousSegment;
				segments[segments.Length - 1].NextSegment = NextSegment;
				return segments;
			}

			public override OcdBaseSegment Reverse() {
				OcdBaseSegment newSegment = Clone(end, control2, control1, start);
				return newSegment;
			}

		}

	}

}
