﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;

namespace OPlanner {

	public partial class CanvasSizeProperties : Window {

		private OCurrentInfo info;

		public Size DefaultSize;

		public CanvasSizeProperties() {
			InitializeComponent();
			DefaultSize = Size.Empty;
		}

		public void SetInfo(OCurrentInfo newInfo) {
			info = newInfo;
			LoadData();
		}

		private void LoadData() {
			if (info.Data == null) {
				return;
			}
			if (info.Data.CanvasSizeMode == AutoManual.Manual) {
				manualPanel.IsEnabled = true;
				manualButton.IsChecked = true;
				manualWidth.Text = Convert.ToString(info.Data.CanvasSize.Width);
				manualHeight.Text = Convert.ToString(info.Data.CanvasSize.Height);
			} else {
				manualPanel.IsEnabled = false;
				manualWidth.Text = Convert.ToString(DefaultSize.Width);
				manualHeight.Text = Convert.ToString(DefaultSize.Height);
				autoButton.IsChecked = true;
			}
		}

		private AutoManual GetSizeMode() {
			if (manualButton.IsChecked.Value) {
				return AutoManual.Manual;
			} else {
				return AutoManual.Auto;
			}
		}

		private void saveButton_Click(object sender, RoutedEventArgs e) {
			SaveData();
			Close();
		}

		private bool Validate(bool showErrors) {
			ErrorManager errorManager = new ErrorManager();
			AutoManual sizeMode = GetSizeMode();
			double num;
			if (sizeMode == AutoManual.Manual) {
				if (manualWidth.Text == "") {
					errorManager.AddError("Width is required");
				} else {
					if (!double.TryParse(manualWidth.Text, out num)) {
						errorManager.AddError("Width must be a number");
					} else {
						if (num < 0) {
							errorManager.AddError("Width must be a positive number");
						}
					}
				}
				if (manualHeight.Text == "") {
					errorManager.AddError("Height is required");
				} else {
					if (!double.TryParse(manualHeight.Text, out num)) {
						errorManager.AddError("Height must be a number");
					} else {
						if (num < 0) {
							errorManager.AddError("Height must be a positive number");
						}
					}
				}
			}
			if (showErrors) {
				errorManager.ShowErrors(this);
			}
			return !errorManager.HasErrors;
		}

		private void SaveData() {
			if (info == null) {
				return;
			}
			if (info.Data == null) {
				return;
			}
			if (!Validate(true)) {
				return;
			}
			bool changed = false;
			AutoManual sizeMode = GetSizeMode();
			double width = 0;
			double height = 0;
			if (sizeMode != info.Data.CanvasSizeMode) {
				changed = true;
			}
			if (sizeMode == AutoManual.Manual) {
				width = double.Parse(manualWidth.Text);
				height = double.Parse(manualHeight.Text);
				if (width != info.Data.CanvasSize.Width) {
					changed = true;
				}
				if (height != info.Data.CanvasSize.Height) {
					changed = true;
				}
			}
			if (changed) {
				info.InvokeDataChanging(this, "Change Canvas Size");
				info.Data.CanvasSizeMode = sizeMode;
				if (sizeMode == AutoManual.Manual) {
					info.Data.CanvasSize.Width = width;
					info.Data.CanvasSize.Height = height;
				}
				info.InvokeDataChanged(this, "Change Canvas Size");
				info.InvokeMapUpdated(this);
			}
		}

		private void cancelButton_Click(object sender, RoutedEventArgs e) {
			Close();
		}

		private void autoButton_Checked(object sender, RoutedEventArgs e) {
			if (info == null) {
				return;
			}
			if (info.Data == null) {
				return;
			}
			manualPanel.IsEnabled = false;
		}

		private void manualButton_Checked(object sender, RoutedEventArgs e) {
			if (info == null) {
				return;
			}
			if (info.Data == null) {
				return;
			}
			manualPanel.IsEnabled = true;
		}

	}
}
